/*
Arduino AJAX IP Webcam
 
 At the very least, along with your Arduino, you'll also need an 
 Ethernet Shield and an Adafruit Serial TTL JPEG Camera (VC0706).
 
 If you can complete the tutorials for both of those items (Ethernet 
 Shield and Serial camera), then you should have no trouble getting 
 this sketch to work. 
 
 The Adafruit Serial JPEG Camera (VC0706) should have the camera's 
 TX connected to pin 2 and the camera's RX to pin 3.
 
 An optional LED can be added to pin 9. 
 
 Also, a temperature sensor, like the LM34 or LM35 can be 
 attached to A0. This is configurable on the "xml_output" page.
 
 Visit TheStuffWeBuild.com for pictures and more.
 
 */

//DebugLevel (in case you are low on flash/ram)
//#define DebugLevel 0 //off. 
//#define DebugLevel 1 //errors only. 
#define DebugLevel 2 //(1)+sd,client requests,taking image. 
//#define DebugLevel 3 //(2)+camera set 
//try shortening long strings to save flash if necessary

//BusyLED used to indicate SD data flow
#define BusyLED 9

//Need these libraries
#include <Adafruit_VC0706.h>
#include <SoftwareSerial.h>        
#include <SPI.h>
#include <Ethernet.h>
#include <SD.h>

//On Uno: camera TX connected to pin 2, camera RX to pin 3:
SoftwareSerial cameraconnection = SoftwareSerial(2, 3);
Adafruit_VC0706 cam = Adafruit_VC0706(&cameraconnection);

//Starting var for sequential file naming
int fname = 0;

//Vars for counting time and time since last pic (tslp)
unsigned long lastpic = 0;
unsigned long tslp = 0;

//Starting var for hit counter
int hits = 0;

//Vars for determining cpu load
unsigned long preload = 0;
int load = 0;

//Max pics to take before rollover to 0 again. Can not exceed 1000
//More pics means "list_dir" requests take longer
int maxpics = 256; 

//Var for setting image resolution. should be 0 for bootup
int cammode = 0;

//Char for making picture file names
static char pfilename[13];

//Used for serving up files on the sd card to client
File webFile;
char* wfilename;

//used for making a root directory list
File entry;

//file name of the default "homepage" file on the card
char rootFileName[] = "index.htm";

//ethernet setup
/////////////////////////////////////////////////////////////////////
// Enter a MAC address and IP address for your controller below.
// The IP address will be dependent on your local network:
byte mac[] = { 
  0x90, 0xA2, 0xDA, 0x00, 0x80, 0xD3 };
IPAddress ip(192,168,1,177);
//IPAddress dns1(192,168,0,1);
//IPAddress gateway(192,168,0,1);
//IPAddress subnet(255,255,255,0);

// Initialize the Ethernet server library
// with the IP address and port you want to use 
// (port 80 is default for HTTP):
EthernetServer server(1220);
/////////////////////////////////////////////////////////////////////

//setup procedure
void setup() {

  //For more precise temperature value.
  analogReference(INTERNAL);

  //Debug led
  pinMode(BusyLED, OUTPUT);

  //Ethernet.begin(mac, ip, dns1, gateway, subnet);
  Ethernet.begin(mac, ip);
  server.begin();
  Serial.begin(9600);

  pinMode(10, OUTPUT);

  //test if SD card is present
  if (!SD.begin(4)) {
#if DebugLevel >= 1
    Serial.print(F("sd failed! "));
#endif
    //return;
  }
  else{
#if DebugLevel >= 2
    Serial.print(F("sd ready. "));
#endif
  }

  //server will try to run regardless, so let serial users know
  Serial.println(F("server running. "));

  // Try to locate the camera and set it up for use
  camcamtest();

  //take a pic to get things primed
  camcam();
}

void loop() {
  int BUFSIZ = 32;
  int index = 0;
  char clientline[BUFSIZ];

  //contently check to see if a client is available
  EthernetClient client = server.available();

  //if you found a client, begin to do server things
  if (client) {
    boolean currentLineIsBlank = true;
    index = 0;

    //if a connection is established, start doing this
    while (client.connected()) {

      //if the client is available, do this
      if (client.available()) {

        //read the client's request
        char c = client.read();
        if (c != '\n' && c != '\r') {
          clientline[index] = c;
          index++;
          if (index >= BUFSIZ)
            index = BUFSIZ -1;
          continue;
        }
        clientline[index] = 0;

        //remember what the client was requesting
        wfilename = processFile(clientline);

#if DebugLevel >= 2
        Serial.print(F("Request: /"));
        Serial.println(wfilename);  
#endif

        /////////////////////////Begin AJAX Requests//////////////////////////////
        //you can add as many as you'd like
        //////////////////////////////////////////////////////////////////////////
        //AJAX request for XML data
        if (strstr(clientline, "GET /?xml") != 0) {
          //you do not need to call 200 header, it is included in dataReq
          dataReq(client);
          break;
        }  

        //AJAX request for taking a low-res picture
        if (strstr(clientline, "GET /?pic1") != 0) {
          cammode = 1;
          code200(client);
          camcamtest();
          camcam();
          client.print(F("done."));
          break;
        }

        //AJAX request for taking a med-res picture
        if (strstr(clientline, "GET /?pic2") != 0) {
          cammode = 2;
          code200(client);
          camcamtest();
          camcam();
          client.print(F("done."));
          break;
        }

        //AJAX request for taking a hi-res picture
        if (strstr(clientline, "GET /?pic3") != 0) {
          cammode = 3;
          code200(client);
          camcamtest();
          camcam();
          client.print(F("done."));
          break;
        }

        //AJAX request for listing root directory
        if (strstr(clientline, "GET /?dir") != 0) {
          code200(client);
          printDirectory(client, 0);
          break;
        }

        //Sample AJAX request
        /*
        if (strstr(clientline, "GET /?egg") != 0) {
         code200(client);
         hits = 12345;
         client.print(F("<META http-equiv=\"refresh\" content=\"0;URL=http://thestuffwebuild.com/instant_starfield.html\"> "));
         break;
         }
         */
         

        /////////////////////////End AJAX Requests//////////////////////////////
        ////////////////////////////////////////////////////////////////////////

        //if no requested file, change wfilename to rootfilenwame
        if (strlen(wfilename) < 2) {
          wfilename = rootFileName;
        }

        //now we send a file. lets start with the http header
        if (SD.exists(wfilename)) {
          code200(client);

          //Turn on debug LED to show we are writing to client
          digitalWrite(BusyLED, HIGH);

          //begin writing requested file to client
          webFile = SD.open(wfilename, FILE_READ);
          if (webFile) {
            while(webFile.available()) {
              client.write(webFile.read());
            }
            delay(400); //may be used/increased if "connection was reset" problems persist with router.
            webFile.close();
          }
          digitalWrite(BusyLED, LOW);
          break;
        }
        else {
          //something happened, send 404 error
          //code404(client);
          code200(client);
          client.print(F("error"));
          break;
        }
        break;
      }
    }

    //close comms
    client.println(F("<!--end-->"));
    delay(1);
    client.flush();
    client.stop(); 
  }

  //calculation for the load counter
  load = millis() - preload;
  preload = millis(); 

}

//Find the filename in the client's header.
char* processFile(char clientline[63]) {
  char *wfilename;
  wfilename = clientline + 5;
  (strstr(clientline, " HTTP"))[0] = 0;
  return wfilename;
}



